/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This file is part of the LibreOffice project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * This file incorporates work covered by the following license notice:
 *
 *   Licensed to the Apache Software Foundation (ASF) under one or more
 *   contributor license agreements. See the NOTICE file distributed
 *   with this work for additional information regarding copyright
 *   ownership. The ASF licenses this file to you under the Apache
 *   License, Version 2.0 (the "License"); you may not use this file
 *   except in compliance with the License. You may obtain a copy of
 *   the License at http://www.apache.org/licenses/LICENSE-2.0 .
 */

#include <drawinglayer/primitive2d/PolyPolygonHatchPrimitive2D.hxx>

#include <drawinglayer/primitive2d/drawinglayer_primitivetypes2d.hxx>
#include <drawinglayer/primitive2d/maskprimitive2d.hxx>
#include <drawinglayer/primitive2d/fillhatchprimitive2d.hxx>
#include <rtl/ref.hxx>
#include <utility>

using namespace com::sun::star;

namespace drawinglayer::primitive2d
{
Primitive2DReference PolyPolygonHatchPrimitive2D::create2DDecomposition(
    const geometry::ViewInformation2D& /*rViewInformation*/) const
{
    if (!getFillHatch().isDefault())
    {
        // create SubSequence with FillHatchPrimitive2D
        const basegfx::B2DRange aPolyPolygonRange(getB2DPolyPolygon().getB2DRange());
        rtl::Reference<FillHatchPrimitive2D> pNewHatch = new FillHatchPrimitive2D(
            aPolyPolygonRange, getDefinitionRange(), getBackgroundColor(), getFillHatch());
        Primitive2DContainer aSubSequence{ pNewHatch };

        // create mask primitive
        return new MaskPrimitive2D(getB2DPolyPolygon(), std::move(aSubSequence));
    }
    return nullptr;
}

PolyPolygonHatchPrimitive2D::PolyPolygonHatchPrimitive2D(
    const basegfx::B2DPolyPolygon& rPolyPolygon, const basegfx::BColor& rBackgroundColor,
    attribute::FillHatchAttribute rFillHatch)
    : maPolyPolygon(rPolyPolygon)
    , maDefinitionRange(rPolyPolygon.getB2DRange())
    , maBackgroundColor(rBackgroundColor)
    , maFillHatch(std::move(rFillHatch))
{
}

PolyPolygonHatchPrimitive2D::PolyPolygonHatchPrimitive2D(basegfx::B2DPolyPolygon aPolyPolygon,
                                                         const basegfx::B2DRange& rDefinitionRange,
                                                         const basegfx::BColor& rBackgroundColor,
                                                         attribute::FillHatchAttribute rFillHatch)
    : maPolyPolygon(std::move(aPolyPolygon))
    , maDefinitionRange(rDefinitionRange)
    , maBackgroundColor(rBackgroundColor)
    , maFillHatch(std::move(rFillHatch))
{
}

bool PolyPolygonHatchPrimitive2D::operator==(const BasePrimitive2D& rPrimitive) const
{
    if (BufferedDecompositionPrimitive2D::operator==(rPrimitive))
    {
        const PolyPolygonHatchPrimitive2D& rCompare
            = static_cast<const PolyPolygonHatchPrimitive2D&>(rPrimitive);

        return (getB2DPolyPolygon() == rCompare.getB2DPolyPolygon()
                && getDefinitionRange() == rCompare.getDefinitionRange()
                && getBackgroundColor() == rCompare.getBackgroundColor()
                && getFillHatch() == rCompare.getFillHatch());
    }

    return false;
}

// provide unique ID
sal_uInt32 PolyPolygonHatchPrimitive2D::getPrimitive2DID() const
{
    return PRIMITIVE2D_ID_POLYPOLYGONHATCHPRIMITIVE2D;
}

} // end drawinglayer::primitive2d namespace

/* vim:set shiftwidth=4 softtabstop=4 expandtab: */
